/* Copyright (C) 2003-2008 Eric Hsiao <erichs0608@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/* Kernel module implementing a ipp2p set type as a bitmap */

#include <linux/module.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/skbuff.h>
#include <linux/errno.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>
#include <linux/spinlock.h>

#include <net/ip.h>

#include <linux/netfilter_ipv4/ip_set_ipp2p.h>

extern int ipp2p_match(const struct sk_buff *skb, const void *matchinfo, const u_int32_t flags);

static int
ipp2p_utest(struct ip_set *set, const void *data, u_int32_t size)
{
	struct ip_set_ipp2p *map = set->data;
	const struct ip_set_req_ipp2p *req = data;
	ip_set_ip_t port = req->ip;

	if (port < map->first_ip || port > map->last_ip)
		return -ERANGE;
	
	DP("set: %s, port: %u", set->name, port);
	return !!test_bit(port - map->first_ip, map->members);	
}

static int
ipp2p_ktest(struct ip_set *set, const struct sk_buff *skb, const u_int32_t *flags)
{
	struct ip_set_ipp2p *map = set->data;

#if 0
    if ( flags[0] & IPSET_MATCH_DST )
    {
        /* only check the protocols that requires wan-2-lan block */
    
    }
#endif

    if ( flags[0] & IPSET_MATCH_REJ && !test_bit(IPP2P_SKYPE, map->members) )
    {
        /* only check the protocols that requires tcp reset */
        return 0;
    }

	return ipp2p_match(skb, map->members, flags[0]);
}


static int
ipp2p_uadd(struct ip_set *set, const void *data, u_int32_t size)
{
	struct ip_set_ipp2p *map = set->data;
	const struct ip_set_req_ipp2p *req = data;
	ip_set_ip_t port = req->ip;

	if (port < map->first_ip || port > map->last_ip)
		return -ERANGE;
	if (test_and_set_bit(port - map->first_ip, map->members))
		return -EEXIST;
	
	DP("set: %s, port %u", set->name, port);
	return 0;	
	
}

static int
ipp2p_kadd(struct ip_set *set, const struct sk_buff *skb, const u_int32_t *flags)
{
	return 0;
}

static int
ipp2p_udel(struct ip_set *set, const void *data, u_int32_t size)
{
	struct ip_set_ipp2p *map = set->data;
	const struct ip_set_req_ipp2p *req = data;
	ip_set_ip_t port = req->ip;

	if (port < map->first_ip || port > map->last_ip)
		return -ERANGE;
	if (!test_and_clear_bit(port - map->first_ip, map->members))
		return -EEXIST;
	
	DP("set: %s, port %u", set->name, port);
	return 0;	
	
}

static int
ipp2p_kdel(struct ip_set *set, const struct sk_buff *skb, const u_int32_t *flags)
{
	return 0;
}


static inline int
__ipp2p_create(const struct ip_set_req_ipp2p_create *req,
		 struct ip_set_ipp2p *map)
{
	if (req->to - req->from > MAX_RANGE) {
		ip_set_printk("range too big, %d elements (max %d)",
			      req->to - req->from + 1, MAX_RANGE+1);
		return -ENOEXEC;
	}
	return bitmap_bytes(req->from, req->to);
}

BITMAP_CREATE(ipp2p)
BITMAP_DESTROY(ipp2p)
BITMAP_FLUSH(ipp2p)

static inline void
__ipp2p_list_header(const struct ip_set_ipp2p *map,
		      struct ip_set_req_ipp2p_create *header)
{
}

BITMAP_LIST_HEADER(ipp2p)
BITMAP_LIST_MEMBERS_SIZE(ipp2p, ip_set_ip_t, (map->last_ip - map->first_ip + 1),
			 test_bit(i, map->members))

static void
ipp2p_list_members(const struct ip_set *set, void *data, char dont_align)
{
	const struct ip_set_ipp2p *map = set->data;
	uint32_t i, n = 0;
	ip_set_ip_t *d;
	
	if (dont_align) {
		memcpy(data, map->members, map->size);
		return;
	}
	
	for (i = 0; i < map->last_ip - map->first_ip + 1; i++)
		if (test_bit(i, map->members)) {
			d = data + n * IPSET_ALIGN(sizeof(ip_set_ip_t));
			*d = map->first_ip + i;
			n++;
		}
}

IP_SET_TYPE(ipp2p, IPSET_DATA_SINGLE)

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Eric Hsiao <erichs0608@gmail.com>");
MODULE_DESCRIPTION("ipp2p type of IP sets");

REGISTER_MODULE(ipp2p)
