#include <stdio.h>
#include <string.h>
#include "./usb.h"
#include <errno.h>
#include   <time.h> 
#include <syslog.h>

/*
 * Temper.c by Robert Kavaler (c) 2009 (relavak.com)
 * All rights reserved.
 *
 * Temper driver for linux. This program can be compiled either as a library
 * or as a standalone program (-DUNIT_TEST). The driver will work with some
 * TEMPer usb devices from RDing (www.PCsensor.com).
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 * 
 * THIS SOFTWARE IS PROVIDED BY Robert Kavaler ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Robert kavaler BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

#include "temper.h"

#define VENDOR_ID  0x1130
#define PRODUCT_ID 0x660c

static int TEMPer2V13ToTemperature(Temper* t, unsigned short word, TemperData* dst) {
#if 0
	word += t->offset; /* calibration value */
#endif

	dst->value = ((float)word) * (125.0 / 32000.0);
	dst->unit = TEMPER_ABS_TEMP;

	return 0;
}
static int TEMPerHUMToTemperature(Temper* t, unsigned short word, TemperData* dst) {
#if 0
	word += t->offset; /* calibration value */
#endif

	/* assuming Vdd = 5V (from USB) and 14bit precision */
	dst->value = ((float)word) * (0.01) + -40.1;
	dst->unit = TEMPER_ABS_TEMP;

	return 0;
}

static int TEMPerHUMToHumidity(Temper* t, unsigned short word, TemperData* dst) {
	const float rh = (float)word;

	/* assuming 12 bits readings */
	const float c1 = -2.0468;
	const float c2 = .0367;
	const float c3 = -1.5955e-6;

	dst->value = c1 + c2*rh + c3*rh*rh;
	dst->unit = TEMPER_REL_HUM;

	return 0;
}

static int TemperUnavailable(Temper* t, unsigned short word, TemperData* dst) {
	dst->value = 0.0;
	dst->unit = TEMPER_UNAVAILABLE;

	return 0;
}


static const struct Product ProductList[] = {
	{
		0x1130, 0x660c,
		"RDing TEMPer",
		{
			TEMPer2V13ToTemperature,
			TEMPer2V13ToTemperature,
		},
	},
	{
		/* Analog Device ADT75 (or similar) based device */
		/* with two temperature sensors (internal & external) */
		0x0c45, 0x7401,
		"RDing TEMPer2V1.3",
		{
			TEMPer2V13ToTemperature,
			TEMPer2V13ToTemperature,
		},
	},
	{
		/* Sensirion SHT1x based device */
		/* with internal humidity & temperature sensor */
		0x0c45, 0x7402,
		"RDing TEMPerHumiV1.1",
		{
			TEMPerHUMToTemperature,
			TEMPerHUMToHumidity,
		},
	},
};
static const unsigned ProductCount = sizeof(ProductList)/sizeof(struct Product);

int TemperInterruptRead(Temper* t, unsigned char *buf, unsigned int len) {
	int ret, i;

	if (t->debug) {
		printf("interrupt read\n");
	}

	ret = usb_interrupt_read(t->handle, 0x82, (char*)buf, len, t->timeout);
	if(t->debug) {
		printf("receiving %d bytes\n",ret);
		for(i = 0; i < ret; ++i) {
			printf("%02x ", buf[i]);
			if ((i+1)%8 == 0) printf("\n");
		}
		printf("\n");
        }

	return ret;
}

int
TemperGetData_new(Temper *t, struct TemperData *data, unsigned int count) {
	unsigned char buf[8];
	int i;
	int ret = TemperInterruptRead(t, buf, sizeof(buf));

	for(i = 0; i < count; ++i) {
		if ((2*i+3) < ret) {
			unsigned short word = ((unsigned char)buf[2*i+2] << 8) | buf[2*i+3];
			t->product->convert[i](t, word, &data[i]);
		}
		else {
			TemperUnavailable(t, 0, &data[i]);
		}
	}
	
	return ret;
}


Temper *
TemperCreate(struct usb_device *dev, int timeout, int debug, const struct Product* product)
{
	Temper *t;
	int ret;

	t = calloc(1, sizeof(*t));
	t->device = dev;
	t->debug = debug;
	t->product = product;
	t->timeout = timeout;
	t->handle = usb_open(t->device);
	if(!t->handle) {
		free(t);
		return NULL;
	}
	if(t->debug) {
		printf("Trying to detach kernel driver\n");
	}

	ret = usb_detach_kernel_driver_np(t->handle, 0);
	if(ret) {
		if(errno == ENODATA) {
			if(t->debug) {
				printf("Device already detached\n");
			}
		} else {
			if(t->debug) {
				printf("Detach failed: %s[%d]\n",
				       strerror(errno), errno);
				printf("Continuing anyway\n");
			}
		}
	} else {
		if(t->debug) {
			printf("detach successful\n");
		}
	}
	ret = usb_detach_kernel_driver_np(t->handle, 1);
	if(ret) {
		if(errno == ENODATA) {
			if(t->debug)
				printf("Device already detached\n");
		} else {
			if(t->debug) {
				printf("Detach failed: %s[%d]\n",
				       strerror(errno), errno);
				printf("Continuing anyway\n");
			}
		}
	} else {
		if(t->debug) {
			printf("detach successful\n");
		}
	}

	if(usb_set_configuration(t->handle, 1) < 0 ||
	   usb_claim_interface(t->handle, 0) < 0 ||
	   usb_claim_interface(t->handle, 1)) {
		usb_close(t->handle);
		free(t);
		return NULL;
	}
	return t;
}

Temper *
TemperCreateFromDeviceNumber(int deviceNum, int timeout, int debug)
{
	struct usb_bus *bus;
	int n, i;

	printf("TemperCreateFromDeviceNumber\n");
	n = 0;
	for(bus=usb_get_busses(); bus; bus=bus->next) {
	    struct usb_device *dev;

	    for(dev=bus->devices; dev; dev=dev->next) {
		if(debug) {
			printf("Found device: %04x:%04x\n",
			       dev->descriptor.idVendor,
			       dev->descriptor.idProduct);
		}
		for(i = 0; i < ProductCount; ++i) {
			if(dev->descriptor.idVendor == ProductList[i].vendor &&
			   dev->descriptor.idProduct == ProductList[i].id) {
				if(debug) {
			    	printf("Found deviceNum %d\n", n);
			    	printf("usb portNum %u\n", dev->config->interface->altsetting->bInterfaceNumber);
				}
				if(n == deviceNum) {
					return TemperCreate(dev, timeout, debug, &ProductList[i]);
				}
				n++;
			}
	    }
	    }
	}
	return NULL;
}

void
TemperFree(Temper *t)
{
	if(t) {
		if(t->handle) {
			usb_close(t->handle);
		}
		free(t);
	}
}

static int
TemperSendCommand(Temper *t, int a, int b, int c, int d, int e, int f, int g, int h)
{
	unsigned char buf[32];
	int ret;

	bzero(buf, 32);
	buf[0] = a;
	buf[1] = b;
	buf[2] = c;
	buf[3] = d;
	buf[4] = e;
	buf[5] = f;
	buf[6] = g;
	buf[7] = h;

	if(t->debug) {
		printf("sending bytes %d, %d, %d, %d, %d, %d, %d, %d\n",
		       a, b, c, d, e, f, g, h);
	}

	ret = usb_control_msg(t->handle, 0x21, 9, 0x200, 0x01,
			    (char *) buf, 32, t->timeout);
	if(ret != 32) {
		perror("usb_control_msg failed");
		return -1;
	}
	return 0;
}

static int
TemperGetData(Temper *t, char *buf, int len)
{
	int ret;

	return usb_control_msg(t->handle, 0xa1, 1, 0x300, 0x01,
			    (char *) buf, len, t->timeout);
}

int
TemperGetTemperatureInC(Temper *t, float *tempC)
{
	FILE *fp;
	char buf[256];
	int ret, temperature, i,stemp[50],tmp[10],time[50];

	printf("TemperGetTemperatureInC start\n");
	TemperSendCommand(t, 10, 11, 12, 13, 0, 0, 2, 0);
	TemperSendCommand(t, 0x54, 0, 0, 0, 0, 0, 0, 0);
	for(i = 0; i < 7; i++) {
		TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
	}
	sleep(1);
	TemperSendCommand(t, 10, 11, 12, 13, 0, 0, 1, 0);
	printf("TemperGetTemperatureInC 222\n");
	ret = TemperGetData(t, buf, 256);
	if(ret < 2) {
		return -1;
	}
	if(buf[2]!=0x31)
	{
		fp=fopen("/tmp/temperlog.log","a+b");
		if(fp)
		{
			fprintf(fp,"buf[2]!=0x31\n");
			fclose(fp);
		}
		return -1;
	}
	memset(stemp,0,50);
	printf("TemperGetTemperatureInC 333\n");
	for(i=0;i<8;i++)
	{
		sprintf(tmp,"%x ",buf[i]);
		strcat(stemp,tmp);
	}
	StdformatLocalTime(time);
	printf("TemperGetTemperatureInC 444\n");
	fp=fopen("/tmp/temperlog.log","a+b");
	if(fp)
	{
		fprintf(fp,"buf=%s %s\n",stemp,time);
		fclose(fp);
		}
        		//syslog(LOG_INFO, "temperature %.2fC %.2fF\n",
	temperature = (buf[1] & 0xFF) + (buf[0] << 8);	
	//temperature += 1152;			// calibration value
	*tempC = temperature * (125.0 / 32000.0);
	return 0;
}

int
TemperGetTemperatureInC_new(Temper *t, float *tempC)
{
	FILE *fp;
	char buf[256];
	int ret, temperature, i,stemp[50],tmp[10],time[50];
	TemperData data[2];
	const unsigned int count = sizeof(data)/sizeof(TemperData);

	printf("TemperGetTemperatureInC_new start\n");
	ret = TemperSendCommand(t,0x01, 0x80, 0x33, 0x01, 0x00, 0x00, 0x00, 0x00 );
	if(ret == -1) {
		printf("TemperGetData  err ret=%d\n", ret);
		return -1;
	}
	//ret = TemperGetData(t, buf, 8);
	ret = TemperGetData_new(t,data, count);
	printf("TemperGetData = %4d\n", ret);
	#if 0
	if(ret != 8) {
		perror("TemperGetData");
		printf("err ret=%d\n", ret);
		return -1;
	}
	#endif
	memset(stemp,0,50);
	//for(i=0;i<8;i++)
	for (i = 0; i < count; ++i)
	{
		printf(";%f %s\n", data[i].value, TemperUnitToString(data[i].unit));
		//sprintf(tmp,"%x ",buf[i]);
		//strcat(stemp,tmp);
		*tempC =data[0].value;
	}
	//if(buf[2]!=0x31)
	#if 0
	if(0)
	{
		fp=fopen("/tmp/temperlog.log","a+b");
		if(fp)
		{
			fprintf(fp,"buf[2]!=0x31\n");
			fclose(fp);
		}
		printf("TemperGetData buf[%s]\n", stemp);
		return -1;
	}
	//printf("TemperGetData buf[%s]\n", stemp);
	StdformatLocalTime(time);
        		//syslog(LOG_INFO, "temperature %.2fC %.2fF\n",
	temperature = (buf[1] & 0xFF) + (buf[0] << 8);	
	//temperature += 1152;			// calibration value
//	*tempC = temperature * (125.0 / 32000.0);
	 int temperatureOut = (buf[5] & 0xff) + (buf[4] << 8);      
	int temperatureIn  = (buf[3] & 0xff) + (buf[2] << 8);	
	 printf("temperatureIn %d\r\n", temperatureIn);	
	 printf("temperatureOut %d\r\n", temperatureOut);
	 *tempC = temperatureIn * (125.0 / 32000.0);
	 #endif
	 
	return 0;
}

int
TemperGetOtherStuff(Temper *t, char *buf, int length)
{
	TemperSendCommand(t, 10, 11, 12, 13, 0, 0, 2, 0);
	TemperSendCommand(t, 0x52, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 10, 11, 12, 13, 0, 0, 1, 0);
	return TemperGetData(t, buf, length);
}

 void StdformatLocalTime(char *pch)
{
	char year[5],month[3],day[3],hour[3],minute[3],secend[3],zero[5];
	struct tm *local;
	//NEMOLOG(stdout,"StdformatLocalTime");
	
	time_t t;
	t=time(NULL);
	local=localtime(&t);

	snprintf(pch,20,"%d-%02d-%02d %02d:%02d:%02d",local->tm_year+1900,local->tm_mon+1,local->tm_mday,local->tm_hour,local->tm_min,local->tm_sec);
	
}
#ifdef UNIT_TEST

#define USB_TIMEOUT 1000	/* milliseconds */

int main(int argc, char* argv[]) 
{
	Temper *t;
	char buf[256],timenow[30],timelast[30],ena[300],*tmp,tmp1[100],warnlog[2000],ftmp[10][20];
	int i, ret,count,unit,syslogen,mailen,smsen,smsprof,ulimit,llimit;
	float tempc,hight=0,lowt=100,avet,tmpc,cal=0;//,ftmp[10];
	FILE *fp;
	
		fp=fopen("/tmp/temperlog.log","a+b");
		if(fp)
		{
			fprintf(fp,"temper start\n");
			fclose(fp);
		}
	//printf("test temper\n");
	//return 0;
	//system("rm /tmp/usbtemperrange.log /tmp/usbtempergraph.log");
	fp=fopen("/etc/config/usb_temper","r");
	if(!fp)
		return 0;
	fread(ena,290,1,fp);
	//printf("ena=%s\n",ena);
	fclose(fp);
	if(!strstr(ena,"option enable\t'1'"))
		return 0;
	
	tmp=strstr(ena,"option temper_unit");
	sscanf(tmp,"option temper_unit\t'%[0-9]'%*",tmp1);
	//printf("temper_unit is %s\n",tmp1);
	unit=atoi(tmp1);
	
	tmp=strstr(ena,"option syslong_en");
	sscanf(tmp,"option syslong_en\t'%[0-9]'%*",tmp1);
	//printf("syslong_en is %s\n",tmp1);
	syslogen=atoi(tmp1);
	
	tmp=strstr(ena,"option mail_en");
	sscanf(tmp,"option mail_en\t'%[0-9]'%*",tmp1);
	//printf("mail_en is %s\n",tmp1);
	mailen=atoi(tmp1);
	
	tmp=strstr(ena,"option sms_en");
	sscanf(tmp,"option sms_en\t'%[0-9]'%*",tmp1);
	//printf("sms_en is %s\n",tmp1);
	smsen=atoi(tmp1);
		
	tmp=strstr(ena,"option sms_profile");
	sscanf(tmp,"option sms_profile\t'%[0-9]'%*",tmp1);
	//printf("sms_profile is %s\n",tmp1);
	smsprof=atoi(tmp1);
	
	tmp=strstr(ena,"option ulimit");
	sscanf(tmp,"option ulimit\t'%[0-9]'%*",tmp1);
	//printf("ulimit is %s\n",tmp1);
	ulimit=atoi(tmp1);
	
	tmp=strstr(ena,"option llimit");
	sscanf(tmp,"option llimit\t'%[0-9]'%*",tmp1);
	//printf("llimit is %s\n",tmp1);
	llimit=atoi(tmp1);

	tmp=strstr(ena,"option cal");
	sscanf(tmp,"option cal\t'%[0-9.-]'",tmp1);
	//printf("cal is %s\n",tmp1);
	cal=atof(tmp1);
	
	usb_set_debug(0);
	usb_init();
	usb_find_busses();
	usb_find_devices();

	//printf("TemperCreateFromDeviceNumber111\n");
	t = TemperCreateFromDeviceNumber(0, USB_TIMEOUT, 1);
	if(!t) {
		perror("TemperCreate");
		exit(-1);
	}


/*	TemperSendCommand(t, 10, 11, 12, 13, 0, 0, 2, 0);
	TemperSendCommand(t, 0x43, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
	TemperSendCommand(t, 0, 0, 0, 0, 0, 0, 0, 0);
*/

	//printf("TemperCreateFromDeviceNumber444\n");
	bzero(buf, 256);
	if( t->product->vendor == VENDOR_ID && t->product->id == PRODUCT_ID){
		ret = TemperGetOtherStuff(t, buf, 256);
		//printf("Other Stuff (%d bytes):\n", ret);
		for(i = 0; i < ret; i++) {
			printf(" %02x", buf[i] & 0xFF);
			if(i % 16 == 15) {
				//printf("\n");
			}
		}
		//printf("\n");

		StdformatLocalTime(timenow);
		//printf("timenow=%s\n",timenow);
		strcpy(timelast,timenow);
		if(TemperGetTemperatureInC(t, &tempc) < 0) {
			perror("TemperGetTemperatureInC");
			return;
		}
	}
	else{
		//new TEMPerV2
		StdformatLocalTime(timenow);
		strcpy(timelast,timenow);
		if(TemperGetTemperatureInC_new(t, &tempc) < 0) {
			perror("TemperGetTemperatureInC");
			return;
		}
	}


	memset(ena,0,300);
	fp=fopen("/tmp/usbtemperrange.log","r+b");
	if(fp)
	{
		fread(ena,290,1,fp);
		//printf("draytek test \n%s\n",ena);
		sscanf(ena,"%[^,]%*s%s\n%[^,]%*s%s\n%[^,]%*s%s\n%[^,]%*s%s\n%s",ftmp[0],ftmp[1],ftmp[2],ftmp[3],ftmp[4],ftmp[5],ftmp[6],ftmp[7],ftmp[8]);
		//printf("%s %s\n%s %s\n%s %s\n%s %s\n%s",ftmp[0],ftmp[1],ftmp[2],ftmp[3],ftmp[4],ftmp[5],ftmp[6],ftmp[7],ftmp[8]);
		hight=atof(ftmp[2]);
		lowt=atof(ftmp[4]);
		avet=atof(ftmp[6]);
		count=atoi(ftmp[8]);

		printf("hight=%f,lowt=%f,avet=%f count=%d\n",hight,lowt,avet,count);
		
		hight=tempc>hight?tempc:hight;
		lowt=tempc<lowt?tempc:lowt;
		avet=count?(avet*count+tempc)/(++count):tempc;
		
		//printf("/tmp/usbtemperrange.log \n%.2f,%.2f\n%.2f,%.2f\n%.2f,%.2f\n%.2f,%.2f\n",ftmp[0],ftmp[1],ftmp[2],ftmp[3],ftmp[4],ftmp[5],ftmp[6],ftmp[7]);

		fclose(fp);
	}
	else{
		hight=lowt=avet=tempc;
		count=1;
		}
	fp=fopen("/tmp/usbtemperrange.log","w+b");
	fprintf(fp,"%.2f, %.2f\n%.2f, %.2f\n%.2f, %.2f\n%.2f, %.2f\n%d",tempc, (9.0 / 5.0 * tempc + 32.0),
	       hight,(9.0 / 5.0 * hight + 32.0),lowt,(9.0 / 5.0 * lowt + 32.0),avet,(9.0 / 5.0 * avet + 32.0),count);
	fclose(fp);
	
	fp=fopen("/tmp/usbtempergraph.log","a+b");
	fprintf(fp,"%.2f,%.2f\n",
	       tempc, (9.0 / 5.0 * tempc + 32.0));
	fclose(fp);
	
	//printf("temperature %.2fC %.2fF\n",
	//       tempc, (9.0 / 5.0 * tempc + 32.0));
	
	if(syslogen)
		syslog(LOG_INFO, "temperature %.2fC %.2fF\n",
	       tempc, (9.0 / 5.0 * tempc + 32.0));
	tmpc=unit?(9.0 / 5.0 * tempc + 32.0):tempc;
	
	if((tmpc+cal>=ulimit)||(tmpc+cal<=llimit))
	{
		sprintf(warnlog," The current temperature is %.2f %c, which is out of the specified normal range (%d - %d %c)!!\n",
			tmpc+cal,unit?'F':'C',llimit,ulimit,unit?'F':'C');
		//fprintf(stdout," The current temperature is %.2f %c, which is out of the specified normal range (%d ~ %d %c)!!\n",
		//	tmpc+cal,unit?'F':'C',llimit,ulimit,unit?'F':'C');
		syslog(LOG_WARNING, "[alarm] temperature %.2f %c\n",tmpc+cal, unit?'F':'C');
		if(mailen)
		{
			fp=fopen("/tmp/temper_mail_body","w+b");
			if(fp){
				fprintf(fp,"%s",warnlog);
				fclose(fp);
				system("/sbin/temper_mail_alert.sh &");
				}
		}
		if(smsen)
		{
			sprintf(warnlog,"/usr/lib/ddns/sms_updater.sh \"%d\" \"The+current+temperature+is+%.2f+%c,+which+is+out+of+the+specified+normal+range+(%d+-+%d+%c)!! &\"",
				smsprof,tmpc+cal,unit?'F':'C',llimit,ulimit,unit?'F':'C');
			//printf("%s",warnlog);
			system(warnlog);
		}
	}
	return 0;
}

#endif
